/*****************************************************************************

	CMDLINE.C
	---------
	
	This module processes the command line arguments. This module is
	used only with Lancaster options active
	
*****************************************************************************/

#include	"estruct.h"
#include	"edef.h"
#include	"etype.h"

#include	<stdio.h>
#if		MSDOS
#include	<getopt.h>
#include	<string.h>
#include	<dos.h>
#endif
#if	BSD
#include	<strings.h>
extern	char	*optarg;
extern	int		optind;
extern	int		opterr;
#endif


#if	LANCASTER
#if USG
#include <string.h>
extern	char	*optarg;
extern	int		optind;
extern	int		opterr;
#endif

/*****************************************************************************

	Local defines
	-------------
	
*****************************************************************************/

#define	MAXFILES	256				/* max names on command line */

struct	FDETAILS					/* details of one file */
		{
			int		read_only;			/* is file read only? */
			char	*name;				/* ptr to file name */
		};


/*****************************************************************************

	Local Procedures
	----------------
	
*****************************************************************************/

static	int		convert_number();
static	void	duplicate_option();
static	void	*get_memory();
static	int		glob_filename();
static	BUFFER	*prepare_file();
static	int		process_file_list();
static	void	set_variable();
static	void	store_filename();
static	void	too_many_files();


/*****************************************************************************

	Local Data
	----------
	
*****************************************************************************/

static	struct	FDETAILS	*file_list,		/* ptr to file details list */
							*file_list_ptr;	/* ptr to next entry in list */
static	int			filecount	= 0;		/* files in list */
				
static	char	*startup_file	= "",		/* startup file name */
				*exe_name;					/* name of executable file */



/*****************************************************************************

	process_options
	---------------
	
	Processes the command line options. This routine requires a "getopt"
	library routine.
	
	bp = process_options(argc,argv,gotoflag,gline,searchflag,cryptflag,ekey)
	
	BUFFER	*bp;				Pointer to first file's buffer
	int		argc;				Argument count
	char	*argv[];			Argument vector
	int		*gotoflag;			Flag for -g
	int		*gline;				Line set by -g
	int		*searchflag;		Flag for -s
	int		*cryptflag;			Flag for -k
	char	*ekey;				Encryption key
	
*****************************************************************************/

BUFFER	*process_options(argc,argv,gotoflag,gline,searchflag,cryptflag,ekey)

int		argc;
char	*argv[];
int		*gotoflag;
int		*gline;
int		*searchflag;
int		*cryptflag;
char	*ekey;

{
	
	int		ch,						/* option character */
			i,						/* work value */
			files,					/* count of files on line or globbed */
			names_given;			/* any names on command line? */
	
	BUFFER	*bp,					/* work pointers */
			*firstbp	= NULL;

	char	*ptr;					/* work pointer */
						
	/* Start by working out our name for use in error messages */

	if ( (exe_name = rindex(argv[0],DIRSEPCHAR)) != NULL )
		exe_name++;
	else
		exe_name	= argv[0];

#if MSDOS
	/* Strip off the ".exe" bit on the end */

	if ( (ptr = rindex(exe_name,'.')) != NULL )
		*ptr	= '\0';

	/* Make sure we have the string in lower case */

	strlwr(exe_name);
#endif

	/* Allocate a buffer to hold details of all the files named */

	file_list	= (struct FDETAILS *)get_memory(sizeof(struct FDETAILS) * MAXFILES);
	file_list_ptr	= file_list;

	/* Now process arguments and options */

	files	= 0;
	names_given	= FALSE;
	
	while (optind != argc)
	{

		/* If the first argument starts with "@", it's the name of a
		*  startup file to be used instead of the default
		*/
		
		if ( *argv[1] == '@' )
		{
			/* It does, so check it */
			
			if ( strlen(argv[1]) == 1 )
			{
				fprintf(stderr,"%s : Invalid startup file name\n",exe_name);
				exit(1);
			}
			else
				startup_file	= argv[1]+1;
				
			/* And advance the argument pointer to start at the next argument */
			
			optind++;
			
		}
		
		/* If the next argument does not start with a "-", we have a list of files
		*  be edited
		*/
		
		if ( *argv[optind] != '-' )
		{
			names_given	= TRUE;
			files += process_file_list(argc,argv,FALSE);
		}
			
		/* And we've either come to the end of the arguments, or met an option
		*  flag
		*/
		
		while ( (ch = getopt(argc,argv,"C:c:G:g:I:i:K:k:RrS:s:V:v:")) != EOF )
		{
			switch ( ch )
			{
					case 'c':				/* List of files to edit */
					case 'C':
								optind--;		/* move back to arg of -e */
								process_file_list(argc,argv,FALSE);
								break;
								
					case 'i':				/* Set a variable */
					case 'I':
					
								set_variable(optarg);
								break;
								
					case 'v':				/* List of read-only files */
					case 'V':
					
								optind--;		/* move back to arg of -v */
								process_file_list(argc,argv,TRUE);
								break;
								
					case 'g':				/* Goto line */
					case 'G':
					
								if ( *gotoflag )
									duplicate_option('g');
									
								*gotoflag	= TRUE;
								if ( (*gline = convert_number(optarg)) == -1 )
								{
									fprintf(stderr,
										"%s :Invalid number follows '-g'\n",
										exe_name);
									exit(1);
								}
								break;
								
#if	CRYPT
					case 'k':				/* Encryption key */
					case 'K':
					
								if ( *cryptflag )
									duplicate_option('k');
									
								*cryptflag = TRUE;
								strcpy(ekey, optarg);
								break;
#endif
	
					case 'r':				/* Restrictive use */
					case 'R':
								restflag = TRUE;
								break;
								
					case 's':				/* Initial search string */
					
								if ( *searchflag != 0 )
									duplicate_option('s');
									
								*searchflag = 1;
								bytecopy(pat,optarg,NPAT);
								setjtable(pat);
								break;
				
					case 'S':				/* Initial EXACT search string */
					
								if ( *searchflag != 0 )
									duplicate_option('s');
									
								*searchflag = 2;
								bytecopy(pat,optarg,NPAT);
								setjtable(pat);
								break;
								
					default:				/* unknown switch */
					
								exit(1);
								
			}
			
		}
	}

	/* If we had any names given on the command line, make sure they
	*  globbed out into something
	*/

	if ( names_given && files == 0 )
	{
		fprintf(stderr,"No files match any wildcard specs supplied\n");
		exit(1);
	}
	
	/* Check for any impossible requests */
	
	if ( *gotoflag && *searchflag != 0 )
	{
		fprintf(stderr,"%s : -s and -g cannot be used together\n",
						exe_name);
		exit(1);
	}
	
	/* We've done all the arguments, so we can initialise the editor */
	
	vtinit();						/* screen */
	if ( eexitflag )
		return(NULL);
	edinit("main");					/* buffers and windows */
	varinit();						/* user variables */
	initchars();					/* character set definitions */
	
	/* And now we can run the startup file */

	startup(startup_file);

	/* And now we can prepare all the file names we've been given. After
	*  we've processed each file we can free the dynamic buffer holding the
	*  name
	*/

	file_list_ptr	= file_list;	
	for ( i = 0; i < filecount; i++ )
	{
		bp = prepare_file(*cryptflag,ekey);
		free(file_list_ptr->name);
		
		/* If this is the first file, note where its buffer is */
		
		if ( i == 0 )
			firstbp	= bp;

		/* And step the list pointer on */

		file_list_ptr++;
	}

	/* With that done we can lose the file list itself */

	free( (char *)file_list );
	
	/* And return the pointer to the first file's buffer */
	
	return(firstbp);	
		
}


				
/*****************************************************************************

	prepare_file
	------------
	
	Prepares the file whose details are pointed by the file list pointer
	
	bp = prepare_file(encrypt,ekey)
	
	BUFFER	*bp;				Pointer to file buffer
	int		encrypt;			TRUE if encrypting
	char	*ekey;				Encryption key
	
*****************************************************************************/

static	BUFFER	*prepare_file(encrypt,ekey)

int		encrypt;
char	*ekey;

{
	BUFFER	*bp;					/* work pointer */
	
	char	bname[NBUFN];			/* buffer name */
	
	/* Create a buffer with a unique name */

	makename(bname,file_list_ptr->name);
	unqname(bname);

	/* Set this to inactive */
				
	bp = bfind(bname, TRUE, 0);
	strcpy(bp->b_fname, file_list_ptr->name);
	bp->b_active = FALSE;

	/* Set the modes appropriatly */
			
	if (file_list_ptr->read_only)
		bp->b_mode |= MDVIEW;

#if	CRYPT
	if (encrypt)
	{
		bp->b_mode |= MDCRYPT;
		crypt((char *)NULL, 0);
		crypt(ekey, strlen(ekey));
		strncpy(bp->b_key, ekey, NPAT);
	}
#endif

	/* And return the buffer pointer */
	
	return(bp);

}


/*****************************************************************************

	process_file_list
	-----------------
	
	Takes successive command line arguments as file names, stopping when
	an argument is found beginning with a "-"
	
	count	= process_file_list(argc,argv,read_only)

	int		count;				Number of names provided
	int		argc;				Argument count
	char	*argv[];			Argument vector
	int		read_only;			TRUE if files are read_only
	
*****************************************************************************/

static	int		process_file_list(argc,argv,read_only)

int		argc;
char	*argv[];
int		read_only;

{
	char	*ptr;						/* work pointer */
	int		wildcard;					/* name wildcarded? */
	
	/* Simply loop till we run out of arguments or meet another option
	*  flag
	*/

	while ( optind != argc && *argv[optind] != '-' )
	{
		/* Scan the name for wildcard characters */

		wildcard	= FALSE;
		ptr	= argv[optind];
				
		while ( *ptr != '\0' )
		{
			if ( *ptr != '*' && *ptr != '?' )
				ptr++;
			else
			{
				wildcard	= TRUE;
				break;
			}
		}

		/* If no wildcard, just store it as it stands; otherwise expand it */

		if ( wildcard )
			return( glob_filename(argv[optind++],read_only) );
		else	
		{
			/* Store as is, if there's room */

			if ( filecount == MAXFILES )
				too_many_files();

			store_filename(argv[optind++],read_only); 
			return(1);
		}
	}	
}


/*****************************************************************************

	duplicate_option
	----------------
	
	Reports the use of a duplicate option
	
	duplicate_option(ch)
	
	int		ch;				Option letter
	
*****************************************************************************/

static	void	duplicate_option(ch)

int		ch;

{
	
	fprintf(stderr,"%s : Option '-%c' may only be given once\n",
							exe_name, ch);
	exit(1);
	
}


/*****************************************************************************

	convert_number
	--------------
	
	Converts a string into a number
	
	value = convert_number(string)

	int		value;					Binary value, or -1 if invalid
	char	*string;				String to convert
	
*****************************************************************************/

static	int	convert_number(string)

char	*string;

{
	
	int		value	= 0;			/* returned value */
	
	while ( *string != 0 )
		if ( *string >= '0' && *string <= '9' )
		{
			value = value * 10 + *string - '0';
			string++;
		}
		else
			return(-1);
			
	return(value);
	
}

/*****************************************************************************

	set_variable
	------------
	
	Sets a variable from the command line. The routine is passed a string
	with the format "varname=value"
	
	set_variable(string)
	
	char	*string;			Input string "varname=value"
	
*****************************************************************************/

static	void	set_variable(string)

char	*string;

{
	VDESC	vd;						/* variable descriptor */
	
	char	*ptr;					/* work value */
	
	/* First make sure we have a "=" in the supplied string */
	
	if ( (ptr = index(string,'=')) == NULL )
	{
		fprintf(stderr,"%s : Invalid variable setting '%s'\n",
							exe_name, string);
		exit(1);
	}
	
	/* We do, so set it to '\0' to separate the two parts and make sure
	*  we actually have a value string
	*/
	
	*ptr++ = '\0';
	if ( *ptr == '\0' )
	{
		fprintf(stderr,"%s : Variable value missing from '%s='\n",
								exe_name, string);
		exit(1);
	}
	
	/* We have the bits. Does the variable exist? */

	findvar(string, &vd, NVSIZE + 1);
	if (vd.v_type == -1)
	{
		fprintf(stderr,"%s : Unknown variable '%s'\n",exe_name,optarg);
		exit(1);
	}
	
	/* It does, so set it */
	
	svar(&vd, ptr);
	
}

/*****************************************************************************

	glob_filename
	-------------

	Expands a wildcarded filename, storing each expanded name in the list
	of files to be processed

	count	= glob_filename(pathname,read_only)

	int		count;						Number of files matching
	char	*pathname;					Wildcarded name
	int		read_only;					File read only?

*****************************************************************************/

static	int		glob_filename(pathname,read_only)

char	*pathname;
int		read_only;

{
#if	MSDOS
	char	drive[3],					/* drive part */
			dir[128],					/* dir part */
			fname[32],					/* name part */
			ext[32],					/* ext part */
			buffer[128];				/* buffer for expanded name */
	int		pfx_size,					/* size of string in front of filename */
			count;						/* number of matches */
	struct	find_t	dirent;				/* directory entry details */

	/* While we have the wildcard name intact, start the directory scan */

	if ( _dos_findfirst(pathname,_A_NORMAL,&dirent) != 0 )
	{
		/* No matches, so just stop here */

		return(0);
	}

	/* Now we can split the name into components and work out the size of
	*  the prefix to put in front of each filename
	*/

	_splitpath(pathname,drive,dir,fname,ext);
	if ( dir[0] != '\0' && dir[strlen(dir)-1] != '\\' )
		strcat(dir,"\\");

	pfx_size	= strlen(drive) + strlen(dir);

	/* And loop to process the names. Starting with the one we've just found
	*  above.
	*/

	for (;;)
	{
		/* Got room in the file list for this one? */

		if ( filecount == MAXFILES )
			too_many_files();
		else
			count++;

		/* Build the full pathname */

		strcpy(buffer,drive);
		strcat(buffer,dir);
		strcat(buffer,strlwr(dirent.name));

		/* Note the full pathname and file mode */

		store_filename(buffer,read_only);
		
		/* Then see if we can get another name, exiting if we can't */

		if ( _dos_findnext(&dirent) != 0 )
			return(count);
	}
#else
	return(0);
#endif
}

/*****************************************************************************

	store_filename
	--------------

	Stores a filename and access mode in the next free entry in the file
	list

	store_filename(name,read_only)

	char	*name;					Ptr to filename
	int		read_only;				File read only?

*****************************************************************************/

static	void	store_filename(name,read_only)

char	*name;
int		read_only;

{
	/* Allocate a buffer to hold the name, copy it over and note the mode */

	file_list_ptr->name	= (char *)get_memory(strlen(name)+1);
	strcpy(file_list_ptr->name,name);
	file_list_ptr->read_only	= read_only;

	/* And step the pointer and counter to the next entry */

	filecount++;
	file_list_ptr++;

}

/*****************************************************************************

	too_many_files
	--------------

	Reports if the number of files given or expanded is too large, and exits

	too_many_files()

*****************************************************************************/

static	void	too_many_files()

{
	fprintf(stderr,"Too many files specified\n");
	exit(1);
}

#endif


/*****************************************************************************

	get_memory
	----------

	Allocates memory, exiting on fail

	ptr	= get_memory(size)

	void	*ptr;				Ptr to memory
	int		size;				Size wanted

*****************************************************************************/

static	void	*get_memory(size)

int		size;

{
	void	*ptr;					/* ptr to block */

	if ( (ptr = (void *)malloc(size)) != NULL )
		return(ptr);
	else
	{
		fprintf(stderr,"Cannot allocate %d bytes of memory\n");
		exit(1);
	}
}
	
