/*
 * The functions in this file handle redisplay. There are two halves, the
 * ones that update the virtual display screen, and the ones that make the
 * physical display screen the same as the virtual display screen. These
 * functions use hints that are left in the windows by the commands.
 *
 * $Header: display.c 2.7 92/03/09 $
 * $Log:	display.c,v $
 * Revision 2.7  92/03/09  11:09:58  phil
 * Changes for PTX
 * 
 * Revision 2.6  90/06/21  15:19:51  dan
 * Ensure successful compilation if LANCASTER != 0
 * 
 * Revision 2.5  90/05/10  11:37:44  dan
 * Yet more tweaks to the minimal update routines in updateline().
 * Could I finally have got it working???
 * 
 * Revision 2.4  90/05/07  17:09:50  dan
 * More kludges for update of reverse-video lines!
 * 
 * Revision 2.3  89/12/15  13:55:01  dan
 * Remove modeline and space-filled line bugs from updateline
 * 
 * Revision 2.2  89/12/07  15:09:08  dan
 * Bug in Alan's vthardscroll removed, reverse video bug in updateline fixed
 * 
 * Revision 2.0.1.2  89/12/06  17:01:59  dan
 * Partially fixed scrolling
 * 
 * Revision 2.0.1.1  89/11/22  15:03:13  dan
 * Variable length pscreen & vscreen lines - first failure!
 * 
 * Revision 2.0  89/11/21  11:20:15  dan
 * IBM PC and other modifications made by Alan Philips
 * 
 * Revision 1.5  89/11/06  14:33:02  dan
 * Add hard-scroll function into term structure
 * 
 * Revision 1.4  89/11/03  17:07:34  dan
 * Add hardware-based smooth scrolling.  Major revisions to reframe() in
 * display.c
 * 
 * Revision 1.3  89/09/04  13:13:18  dan
 * Move TTkclose call to BEFORE TTclose, as the former may need access to
 * the terminal.
 * 
 * Revision 1.2  89/08/31  16:11:18  dan
 * Modify tab handling again - Now works as follows:
 * Each buffer has it's own tab sizes.
 * Global variables tabsize and stabsize are the default values used for
 * new buffers.
 * Setting $hardtabs and $softtabs only changes the default.
 * <n> handle-tab sets the tabs for the current buffer.  Use -ve n
 * to set hard tabs rather than soft tabs.
 * 
 */

#include	<stdio.h>
#include	"estruct.h"
#include	"etype.h"
#include	"edef.h"
#include	"elang.h"

typedef struct	VIDEO {
	int	v_flag; 		/* Flags */
#if	COLOR
	int	v_fcolor;		/* current forground color */
	int	v_bcolor;		/* current background color */
	int	v_rfcolor;		/* requested forground color */
	int	v_rbcolor;		/* requested background color */
#endif
	unsigned char	v_size;		/* length of non-space part of line */	
	char	v_text[1];		/* Screen data. */
}	VIDEO;

#define VFCHG	0x0001			/* Changed flag 		*/
#define VFEXT	0x0002			/* extended (beyond column 80)	*/
#define VFREV	0x0004			/* reverse video status 	*/
#define VFREQ	0x0008			/* reverse video request	*/
#define VFCOL	0x0010			/* color change requested	*/

static VIDEO   **vscreen;		       /* Virtual screen. */
#if	HARDSCRL
static	VIDEO	**vstemp;
#endif
#if	MEMMAP == 0
static VIDEO   **pscreen;		       /* Physical screen. */
#if	HARDSCRL
static	VIDEO	**pstemp;
#endif
#endif

/*
 * Initialize the data structures used by the display code. The edge vectors
 * used to access the screens are set up. The operating system's terminal I/O
 * channel is set up. All the other things get initialized at compile time.
 * The original window has "WFCHG" set, so that it will get completely
 * redrawn on the first call to "update".
 */
PASCAL NEAR vtinit()
{
    register int i;
    register VIDEO *vp;

    TTopen();		/* open the screen */
    TTkopen();		/* open the keyboard */
    TTrev(FALSE);
    vscreen = (VIDEO **) malloc(term.t_mrow*sizeof(VIDEO *));
#if	HARDSCRL
	vstemp = (VIDEO **) malloc(term.t_mrow*sizeof(VIDEO *));
#endif

    if (vscreen == NULL)
	meexit(1);

#if	MEMMAP == 0
    pscreen = (VIDEO **) malloc(term.t_mrow*sizeof(VIDEO *));

#if	HARDSCRL
    pstemp = (VIDEO **) malloc(term.t_mrow*sizeof(VIDEO *));
#endif

#if	HARDSCRL
    if (pscreen == NULL || pstemp == NULL)
#else
    if (pscreen == NULL)
#endif
	meexit(1);
#endif

    for (i = 0; i < term.t_mrow; ++i)
	{
	vp = (VIDEO *) malloc(sizeof(VIDEO)+term.t_mcol);

	if (vp == NULL)
	    meexit(1);

	vp->v_flag = 0;
#if	COLOR
	vp->v_rfcolor = 7;
	vp->v_rbcolor = 0;
#endif
	vscreen[i] = vp;
#if	MEMMAP == 0
	vp = (VIDEO *) malloc(sizeof(VIDEO)+term.t_mcol);

	if (vp == NULL)
	    meexit(1);

	vp->v_flag = 0;
	pscreen[i] = vp;
#endif
	}
}

#if	CLEAN
/* free up all the dynamically allocated video structures */

PASCAL NEAR vtfree()
{
	int i;
	for (i = 0; i < term.t_mrow; ++i) {
		free(vscreen[i]);
#if	MEMMAP == 0
		free(pscreen[i]);
#endif
	}
	free(vscreen);
#if	HARDSCRL
	free(vstemp);
#endif
#if	MEMMAP == 0
	free(pscreen);
#if	HARDSCRL
	free(pstemp);
#endif
#endif
}
#endif

/*
 * Clean up the virtual terminal system, in anticipation for a return to the
 * operating system. Move down to the last line and clear it out (the next
 * system prompt will be written in the line). Shut down the channel to the
 * terminal.
 */
PASCAL NEAR vttidy()
{
    mlerase();
    movecursor(term.t_nrow, 0);
#if	LANCASTER
    TTkclose();
    TTflush();
    TTclose();
#else
    TTflush();
    TTclose();
    TTkclose();
#endif
}

/*
 * Set the virtual cursor to the specified row and column on the virtual
 * screen. There is no checking for nonsense values; this might be a good
 * idea during the early stages.
 */
PASCAL NEAR vtmove(row, col)
{
    vtrow = row;
    vtcol = col;
}

/* Write a character to the virtual screen. The virtual row and
   column are updated. If we are not yet on left edge, don't print
   it yet. If the line is too long put a "$" in the last column.
   This routine only puts printing characters into the virtual
   terminal buffers. Only column overflow is checked.
*/

#if LANCASTER
PASCAL NEAR vtputc(c,tabsize)
int tabsize;
#else
PASCAL NEAR vtputc(c,dummy_arg)
int dummy_arg;
#endif
int c;

{
	register VIDEO *vp;	/* ptr to line being updated */

	vp = vscreen[vtrow];

	if (c == '\t') {
		do {
			vtputc(' ',tabsize);
		} while (((vtcol + taboff) % (tabsize)) != 0);
	} else if (vtcol >= term.t_ncol) {
		++vtcol;
		vp->v_text[term.t_ncol - 1] = '$';
	} else if (c < 0x20 || c == 0x7F) {
		vtputc('^',tabsize);
		vtputc(c ^ 0x40,tabsize);
	} else {
		if (vtcol >= 0)
			vp->v_text[vtcol] = c;
		++vtcol;
	}
}

/*
 * Erase from the end of the software cursor to the end of the line on which
 * the software cursor is located.
 */
PASCAL NEAR vteeol()
{
    register VIDEO	*vp;

	vp = vscreen[vtrow];
	if (vp->v_flag & (VFREV | VFREQ | VFCOL))
	{
		/* It's safer to pad reverse video lines with spaces */
	    vp = vscreen[vtrow];
    	while (vtcol < term.t_ncol)
			vp->v_text[vtcol++] = ' ';
	}
	
	if ( vtcol <= term.t_ncol )
		vscreen[vtrow]->v_size	= vtcol;
	else
		vscreen[vtrow]->v_size	= term.t_ncol;
}

/* upscreen:	user routine to force a screen update
		always finishes complete update 	*/

PASCAL NEAR upscreen(f, n)

{
	update(TRUE);
	return(TRUE);
}

/*
 * Make sure that the display is right. This is a three part process. First,
 * scan through all of the windows looking for dirty ones. Check the framing,
 * and refresh the screen. Second, make sure that "currow" and "curcol" are
 * correct for the current window. Third, make the virtual and physical
 * screens the same.
 */
PASCAL NEAR update(force)

int force;	/* force update past type ahead? */

{
	register WINDOW *wp;

#if	TYPEAH
	if (force == FALSE && typahead())
#if TEST
	{
		dprint("Typeahead detected in update\n");
#endif
		return(TRUE);
#if TEST
	}
#endif
#endif
#if	VISMAC == 0
	if (force == FALSE && kbdmode == PLAY)
		return(TRUE);
#endif

	/* update any windows that need refreshing */
	wp = wheadp;
	while (wp != NULL) {
		if (wp->w_flag) {
			/* if the window has changed, service it */
			reframe(wp);	/* check the framing */
			if ((wp->w_flag & ~WFMODE) == WFEDIT)
				updone(wp);	/* update EDITed line */
			else if (wp->w_flag & ~WFMOVE)
				updall(wp);	/* update all lines */
			if (wp->w_flag & WFMODE)
				modeline(wp);	/* update modeline */
			wp->w_flag = 0;
			wp->w_force = 0;
		}
		/* on to the next window */
		wp = wp->w_wndp;
	}

	/* recalc the current hardware cursor location */
	updpos();

#if	MEMMAP
	/* update the cursor and flush the buffers */
	movecursor(currow, curcol - lbound);
#endif

	/* check for lines to de-extend */
	upddex();

	/* if screen is garbage, re-plot it */
	if (sgarbf != FALSE)
		updgar();

	/* update the virtual screen to the physical screen */
	updupd(force);

	/* update the cursor and flush the buffers */
	movecursor(currow, curcol - lbound);
	TTflush();

	return(TRUE);
}

/*	reframe:	check to see if the cursor is on in the window
			and re-frame it if needed or wanted		*/

PASCAL NEAR reframe(wp)

WINDOW *wp;

{
	register LINE *lp;	/* search pointer */
	register LINE *rp;	/* reverse search pointer */
	register LINE *hp;	/* ptr to header line in buffer */
	register LINE *tp;	/* temp debugging pointer */
	register int i;		/* general index/ number of lines between cursor */
						/* and the new top line */
	register int nlines;	/* number of lines in current window */
	int nscroll;	/* Number of lines to hard scroll */
	int scrl_marg;	/* Scrolling margin */

	nscroll = 0;

	/* figure out our window size */
	nlines = wp->w_ntrows;
	if (modeflag == FALSE)
		nlines++;

	/* Work out the scroll margin */

	scrl_marg = abs(sscroll);
	if (!scrl_marg || scrl_marg > nlines/2) scrl_marg=nlines/2;

	/* The screen will be re-framed so that the point is "i" lines down 	*/
	/* from the top.  nscroll is set to the number of lines (+ve or -ve) 	*/
	/* that this will displace the screen */
	
	if ( wp->w_dotp!=wp->w_bufp->b_linep && (wp->w_flag & WFFORCE) == 0 )
	{
		/* search backwards and forwards from top-of-screen in buffer 		*/
		/* for the point to determine i and nscroll							*/

		tp = lp = rp = wp->w_linep;
		hp = wp->w_bufp->b_linep;

		while ( nscroll<=(nlines+scrl_marg) && ( (lp != hp) || (rp != hp) ) )
		{

			if (lp == wp->w_dotp)
			{
				if ( sscroll<0 && nscroll<scrl_marg)
				{
					/* dot was above top scroll margin - scroll up file */
					i = scrl_marg-1;
					nscroll -= i;
					break;
				}
				if ( nscroll > nlines - ((sscroll<0) ? scrl_marg : 1) )
				{
					/* dot is below bottom scroll margin - scroll down file */		
					i = nlines - scrl_marg;
					nscroll -= i;
					break;
				}
				/* No scroll required */
				return(TRUE);
			}

			/* Dot found above current screen */

			if (rp == wp->w_dotp) {
				i = scrl_marg-1;
				nscroll = -nscroll-i;
				break;
			}

			/* advance forward and back */
			if (lp != hp)
				lp = lforw(lp);
			if (lback(rp) != hp)
				rp = lback(rp);

			nscroll++;
			
			/* problems????? */
/*			if (lp == tp || rp == tp) {
				printf("BUG IN SMOOTH SCROLL--GET DAN!\n");
				TTgetc();
			} */
		}
 
		if (!sscroll || nscroll > (nlines/2))
		{
			/* Centre current line on screen, wherever it is */
			i = nlines/2;
			nscroll = 0;
		}
	} else {

		/* Forced redraw */

		i = wp->w_force;
		
		if (i > 0) 
		{	
			/* only one screen worth of lines max */
			if (--i >= nlines)
 				i = nlines - 1;
		}
		else 
			if (i < 0) 
			{	
				/* negative update???? */
				i += nlines;
				if (i < 0)
					i = 0;
			} else
				i = nlines / 2
		;
	}
	
	/* backup to new line at top of window */
	lp = wp->w_dotp;
	while (i != 0 && lback(lp) != wp->w_bufp->b_linep) {
		--i;
		if (i < 0) {
			printf("OTHER BUG IN DISPLAY --- GET DAN!!!\n");
			TTgetc();
			break;
		}
		lp = lback(lp);
 	}

	/* and reset the current line at top of window */

	if(lp == wp->w_linep && !(wp->w_flag & WFFORCE) )
		/* No change in screen position - hit top of buffer */
		return(TRUE);
		
	wp->w_linep = lp;

	/* If the screen has been displaced, scroll it or mark it for update */

	if ( !(wp->w_flag & WFHARD) )
	{
#if HARDSCRL
		if ( nscroll && term.t_scroll && (!i) && ( abs(nscroll) <= nlines/2) )
			vthardscroll(wp,nscroll);
		else
			wp->w_flag |= WFHARD;
#else
		wp->w_flag |= WFHARD;
#endif
	}
	wp->w_flag &= ~WFFORCE;
	return(TRUE);
}

/*	updone: update the current line to the virtual screen		*/

PASCAL NEAR updone(wp)

WINDOW *wp;	/* window to update current line in */

{
	register LINE *lp;	/* line to update */
	register int sline;	/* physical screen line to update */
	register int i;
#if LANCASTER
	int tabsize;
	tabsize=wp->w_bufp->b_tabsize;
#endif
	/* search down the line we want */
	lp = wp->w_linep;
	sline = wp->w_toprow;
	while (lp != wp->w_dotp) {
		++sline;
		lp = lforw(lp);
	}

	/* and update the virtual line */
	vscreen[sline]->v_flag |= VFCHG;
	vscreen[sline]->v_flag &= ~VFREQ;
	taboff = wp->w_fcol;
	vtmove(sline, -taboff);
	for (i=0; i < llength(lp); ++i)
		vtputc(lgetc(lp, i),tabsize);
#if	COLOR
	vscreen[sline]->v_rfcolor = wp->w_fcolor;
	vscreen[sline]->v_rbcolor = wp->w_bcolor;
#endif
	vteeol();
	taboff = 0;
}

/*	updall: update all the lines in a window on the virtual screen */

PASCAL NEAR updall(wp)

WINDOW *wp;	/* window to update lines in */

{
	register LINE *lp;	/* line to update */
	register int sline;	/* physical screen line to update */
	register int i;
	register int nlines;	/* number of lines in the current window */
#if LANCASTER
	int tabsize;
	tabsize=wp->w_bufp->b_tabsize;
#endif
	/* search down the lines, updating them */
	lp = wp->w_linep;
	sline = wp->w_toprow;
	nlines = wp->w_ntrows;
	if (modeflag == FALSE)
		nlines++;
	taboff = wp->w_fcol;
	while (sline < wp->w_toprow + nlines) {

		/* and update the virtual line */
		vscreen[sline]->v_flag |= VFCHG;
		vscreen[sline]->v_flag &= ~VFREQ;
		vtmove(sline, -taboff);
		if (lp != wp->w_bufp->b_linep) {
			/* if we are not at the end */
			for (i=0; i < llength(lp); ++i)
				vtputc(lgetc(lp, i),tabsize);
			lp = lforw(lp);
		}

		/* make sure we are on screen */
		if (vtcol < 0)
			vtcol = 0;

		/* on to the next one */
#if	COLOR
		vscreen[sline]->v_rfcolor = wp->w_fcolor;
		vscreen[sline]->v_rbcolor = wp->w_bcolor;
#endif
		vteeol();
		++sline;
	}
	taboff = 0;
}

#if HARDSCRL

/*******************************************************************************
	vthardscroll

	A window in the physical screen is to be hardware scrolled. Scroll the
	corresponding lines in the virtual terminal using memory moves, updating
	only the newly exposed lines. Relies on compiler
	supporting structure assignments.
*******************************************************************************/

PASCAL NEAR vthardscroll(wp,nscroll)

WINDOW *wp;		/* window to update lines in */
int nscroll;	/* number of lines to scroll */

{
	register LINE *lp;	/* line to update */
	register int sline;	/* physical screen line to update */
	register int i;		/* General "for" index */
	register int nlines;	/* number of lines in the current window */
	register int scrdown;	/* non-zero if scrolling down */
	register int limit;		/* Holds limit for various loops */
	register	int	old,new;
	register	VIDEO	*temp;
		
	/* Get window details */
#if LANCASTER		
	int tabsize;
	tabsize=wp->w_bufp->b_tabsize;
#endif
	lp = wp->w_linep;
	nlines = wp->w_ntrows;
	taboff = wp->w_fcol;

	/* Do the hardware scroll */

#if	COLOR
	TTscroll(wp->w_toprow,wp->w_toprow+nlines-1,nscroll,
					wp->w_fcolor, wp->w_bcolor);
#else
	TTscroll(wp->w_toprow,wp->w_toprow+nlines-1,nscroll);
#endif
	
	/* I can go up, or down... */

	if ( nscroll<0 )
	{
		scrdown = TRUE; 
		nscroll = -nscroll;
	}
	else
		scrdown = FALSE;

	
	/* Part of the update can be done just by copying pointers around in	*/
	/* the virtual screen.													*/
	
	if ( scrdown )
	{
		/* Save the pointers to the lines that we've scrolled off the screen */
		
		sline	= wp->w_toprow + nlines - nscroll;
		for ( i = 0; i < nscroll; i++, sline++ )
		{
			vstemp[i]	= vscreen[sline];
#if	MEMMAP	== 0
			pstemp[i]	= pscreen[sline];
#endif
		}
		
		/* Then shuffle the remaining pointers down, making sure we leave
		*  the mode line untouched
		*/
		
		old		= wp->w_toprow + nlines - nscroll - 1;
		new		= wp->w_toprow + nlines - 1;
		
		for ( i = 0; i < nlines - nscroll; i++ )
		{
			vscreen[new]	= vscreen[old];
#if	MEMMAP == 0
			pscreen[new]	= pscreen[old];
#endif
			old--;
			new--;
		}
		
		/* And finally put the saved pointers back at the top of this
		*  window
		*/
		
		new	= wp->w_toprow;
		
		for ( i = 0; i < nscroll; i++, new++ )
		{
			vscreen[new]	= vstemp[i];
#if	MEMMAP	== 0
			pscreen[new]	= pstemp[i];
#endif
		}    
		sline = wp->w_toprow;
	}
	else		
	{
		/* Save the pointers to the lines that we've scrolled off the screen */
		
		sline	= wp->w_toprow;
		for ( i = 0; i < nscroll; i++, sline++ )
		{
			vstemp[i]	= vscreen[sline];
#if	MEMMAP	== 0
			pstemp[i]	= pscreen[sline];
#endif
		}
		
		/* Then shuffle the remaining pointers up, making sure we leave
		*  the mode line untouched
		*/
		
		old		= wp->w_toprow + nscroll;
		new		= wp->w_toprow;
		
		for ( i = 0; i < nlines - nscroll; i++ )
		{
			vscreen[new]	= vscreen[old];
#if	MEMMAP == 0
			pscreen[new]	= pscreen[old];
#endif
			if (lp != wp->w_bufp->b_linep)
				lp	= lforw(lp);

			old++;
			new++;
		}
		
		/* And finally put the saved pointers back at the bottom of this
		*  window
		*/
		
		new	= wp->w_toprow + nlines - nscroll;
		
		for ( i = 0; i < nscroll; i++, new++ )
		{
			vscreen[new]	= vstemp[i];
#if	MEMMAP	== 0
			pscreen[new]	= pstemp[i];
#endif
		}  
		
	}	

	/* The lines that are now empty on the actual hardware display need to
	*  be rewritten
	*/

	if ( scrdown )
	{
		/* Screen scrolled down, so blank lines are at the top of the
		*  window
		*/

		sline	= wp->w_toprow;
	}
	else
	{
		/* Screen scrolled up, so blank lines are at the bottom of the
		*  window
		*/

		sline	= wp->w_toprow + nlines - nscroll;
	}

	for ( new = 0; new < nscroll; new++ )
	{
		/* Update the virtual line, and mark the physical screen as garbage */
		
		vscreen[sline]->v_flag |= VFCHG;
#if	MEMMAP == 0
		pscreen[sline]->v_flag |= VFCHG;
#endif
		vscreen[sline]->v_flag &= ~VFREQ;
		vtmove(sline, -taboff);
		if (lp != wp->w_bufp->b_linep)
		{
			/* if we are not at the end */
			
			for (i=0; i < llength(lp); ++i)
				vtputc(lgetc(lp, i),tabsize);
				
			lp = lforw(lp);
		}

		/* Make sure we are on screen */
		
		if (vtcol < 0)
			vtcol = 0;

		/* On to the next one */
#if	COLOR
		vscreen[sline]->v_rfcolor = wp->w_fcolor;
		vscreen[sline]->v_rbcolor = wp->w_bcolor;
#endif
		vteeol();
		++sline;
	}
	
	taboff = 0;
}

#endif

/*	updpos: update the position of the hardware cursor and handle extended
		lines. This is the only update for simple moves.	*/

PASCAL NEAR updpos()

{
	register LINE *lp;
	register int c;
	register int i;

	/* find the current row */
	lp = curwp->w_linep;
	currow = curwp->w_toprow;
	while (lp != curwp->w_dotp) {
		++currow;
		lp = lforw(lp);
	}

	/* find the current column */
	curcol = 0;
	i = 0;
	while (i < curwp->w_doto) {
		c = lgetc(lp, i++);
		if (c == '\t')
#if LANCASTER
			curcol += - (curcol % curbp->b_tabsize) + (curbp->b_tabsize - 1);
#else
 			curcol += - (curcol % tabsize) + (tabsize - 1);
#endif
		else
			if (c < 0x20 || c == 0x7f)
				++curcol;

		++curcol;
	}

	/* adjust by the current first column position */
	curcol -= curwp->w_fcol;

	/* make sure it is not off the left side of the screen */
	while (curcol < 0) {
		if (curwp->w_fcol >= hjump) {
			curcol += hjump;
			curwp->w_fcol -= hjump;
		} else {
			curcol += curwp->w_fcol;
			curwp->w_fcol = 0;
		}
		curwp->w_flag |= WFHARD | WFMODE;
	}

	/* if horizontall scrolling is enabled, shift if needed */
	if (hscroll) {
		while (curcol >= term.t_ncol - 1) {
			curcol -= hjump;
			curwp->w_fcol += hjump;
			curwp->w_flag |= WFHARD | WFMODE;
		}
	} else {
	/* if extended, flag so and update the virtual line image */
		if (curcol >=  term.t_ncol - 1) {
			vscreen[currow]->v_flag |= (VFEXT | VFCHG);
			updext();
		} else
			lbound = 0;
	}

	/* update the current window if we have to move it around */
	if (curwp->w_flag & WFHARD)
		updall(curwp);
	if (curwp->w_flag & WFMODE)
		modeline(curwp);
	curwp->w_flag = 0;
}

/*	upddex: de-extend any line that derserves it		*/

PASCAL NEAR upddex()

{
	register WINDOW *wp;
	register LINE *lp;
	register int i,j;
	register int nlines;	/* number of lines in the current window */
#if LANCASTER
	int tabsize;
#endif

	wp = wheadp;

	while (wp != NULL) {
		lp = wp->w_linep;
		i = wp->w_toprow;
		nlines = wp->w_ntrows;
#if LANCASTER
		tabsize=wp->w_bufp->b_tabsize;
#endif
		if (modeflag == FALSE)
			nlines++;

		while (i < wp->w_toprow + nlines) {
			if (vscreen[i]->v_flag & VFEXT) {
				if ((wp != curwp) || (lp != wp->w_dotp) ||
				   (curcol < term.t_ncol - 1)) {
					taboff = wp->w_fcol;
					vtmove(i, -taboff);
					for (j = 0; j < llength(lp); ++j)
						vtputc(lgetc(lp, j),tabsize);
					vteeol();
					taboff = 0;

					/* this line no longer is extended */
					vscreen[i]->v_flag &= ~VFEXT;
					vscreen[i]->v_flag |= VFCHG;
				}
			}
			lp = lforw(lp);
			++i;
		}
		/* and onward to the next window */
		wp = wp->w_wndp;
	}
}

/*	updgar: if the screen is garbage, clear the physical screen and
		the virtual screen and force a full update		*/

PASCAL NEAR updgar()

{
	register int i;
#if	MEMMAP == 0
	register int j;
	register char *txt;
#endif

	for (i = 0; i < term.t_nrow; ++i) {
		vscreen[i]->v_flag |= VFCHG;
#if	REVSTA
		vscreen[i]->v_flag &= ~VFREV;
#endif
#if	COLOR
		vscreen[i]->v_fcolor = gfcolor;
		vscreen[i]->v_bcolor = gbcolor;
#endif
#if	MEMMAP == 0
		pscreen[i]->v_size = 0;
#endif
	}

	movecursor(0, 0);		 /* Erase the screen. */
	(*term.t_eeop)();
	sgarbf = FALSE; 		 /* Erase-page clears */
	mpresf = FALSE; 		 /* the message area. */
#if	COLOR
	mlerase();			/* needs to be cleared if colored */
#endif
}

/*	updupd: update the physical screen from the virtual screen	*/

PASCAL NEAR updupd(force)

int force;	/* forced update flag */

{
	register VIDEO *vp1;
	register int i;
	
	for (i = 0; i < term.t_nrow; ++i) {
		vp1 = vscreen[i];

		/* for each line that needs to be updated*/
		if ((vp1->v_flag & VFCHG) != 0) {
#if	TYPEAH
			if (force == FALSE && typahead())
#if TEST
			{
				dprint("Typeahead detected in updupd\n");
#endif
				return(TRUE);
#if TEST
			}
#endif
#endif
#if	MEMMAP
			updateline(i, vp1);
#else
			updateline(i, vp1, pscreen[i]);
#endif
		}
	}
	return(TRUE);
}

/*	updext: update the extended line which the cursor is currently
		on at a column greater than the terminal width. The line
		will be scrolled right or left to let the user see where
		the cursor is
								*/
PASCAL NEAR updext()

{
	register int rcursor;	/* real cursor location */
	register LINE *lp;	/* pointer to current line */
	register int j; 	/* index into line */

	/* calculate what column the real cursor will end up in */
	rcursor = ((curcol - term.t_ncol) % term.t_scrsiz)
			+ term.t_margin;
	lbound = curcol - rcursor + 1;
	taboff = lbound + curwp->w_fcol;

	/* scan through the line outputing characters to the virtual screen */
	/* once we reach the left edge					*/
	vtmove(currow, -taboff); /* start scanning offscreen */
	lp = curwp->w_dotp;		/* line to output */
	for (j=0; j<llength(lp); ++j)	/* until the end-of-line */
#if LANCASTER
		vtputc(lgetc(lp, j),curbp->b_tabsize);
#else
		vtputc(lgetc(lp, j));
#endif
	/* truncate the virtual line, restore tab offset */
	vteeol();
	taboff = 0;

	/* and put a '$' in column 1 */
	vscreen[currow]->v_text[0] = '$';
}

/*
 * Update a single line. This does not know how to use insert or delete
 * character sequences; we are using VT52 functionality. Update the physical
 * row and column variables. It does try an exploit erase to end of line. The
 * RAINBOW version of this routine uses fast video.
 */
#if	MEMMAP
/*	UPDATELINE specific code for the IBM-PC and other compatables */

PASCAL NEAR updateline(row, vp1)

int row;		/* row of screen to update */
struct VIDEO *vp1;	/* virtual screen image */

{
#if	COLOR
	scwrite(row, vp1->v_text, vp1->v_rfcolor, vp1->v_rbcolor, vp1->v_size);
	vp1->v_fcolor = vp1->v_rfcolor;
	vp1->v_bcolor = vp1->v_rbcolor;
#else
	if (vp1->v_flag & VFREQ)
		scwrite(row, vp1->v_text, 0, 7, vp1->v_text);
	else
		scwrite(row, vp1->v_text, 7, 0, vp1->v_text);
#endif
	vp1->v_flag &= ~(VFCHG | VFCOL);	/* flag this line as changed */

}

#else

PASCAL NEAR updateline(row, vp1, vp2)

int row;		/* row of screen to update */
struct VIDEO *vp1;	/* virtual screen image */
struct VIDEO *vp2;	/* physical screen image */

{
#if RAINBOW
/*	UPDATELINE specific code for the DEC rainbow 100 micro	*/

    register char *cp1;
    register char *cp2;
    register int nch;

    /* since we don't know how to make the rainbow do this, turn it off */
    flags &= (~VFREV & ~VFREQ);

    cp1 = &vp1->v_text[0];		      /* Use fast video. */
    cp2 = &vp2->v_text[0];
    putline(row+1, 1, cp1);
    nch = term.t_ncol;

    do
	{
	*cp2 = *cp1;
	++cp2;
	++cp1;
	}
    while (--nch);
    *flags &= ~VFCHG;
#else
/*	UPDATELINE code for all other versions		*/

	register char *cp1;
	register char *cp2;
	register char *cp3;
	register char *cp4;
	register char *cp5;
	register int nbflag;	/* non-blanks to the right flag? */
	int rev;		/* reverse video flag */
	int req;		/* reverse video request flag */
	int upcol;		/* update column (KRS) */
	int vissize;	/* length of nonblank line visible in screen window */
	int overlap;	/* length of overlap between physical & virtual line */
	
	/* Line length is min(vp1->v_size,term.t_ncol) */
	
	vissize = (vp1->v_size < term.t_ncol) ? vp1->v_size : term.t_ncol;
	overlap = (vissize < vp2->v_size) ? vissize : vp2->v_size;

	/* set up pointers to virtual and physical lines */
	cp1 = &vp1->v_text[0];
	cp2 = &vp2->v_text[0];

#if	COLOR
	TTforg(vp1->v_rfcolor);
	TTbacg(vp1->v_rbcolor);
#endif

#if	HARDSCRL | REVSTA | COLOR
	/* if we need to change the reverse video status of the
	   current line, we need to re-write the entire line	 */
	rev = (vp1->v_flag & VFREV) == VFREV;
	req = (vp1->v_flag & VFREQ) == VFREQ;
	if ((rev != req) || rev

#if	COLOR
	    || (vp1->v_fcolor != vp1->v_rfcolor) || (vp1->v_bcolor != vp1->v_rbcolor)
#endif
#if	HP150
	/* the HP150 has some reverse video problems */
	    || req || rev
#endif
#if HARDSCRL
		|| (vp2->v_flag & VFCHG)	/* Phys screen line trashed by scroll */
#endif
	) {
		movecursor(row, 0);	/* Go to start of line. */
		/* set rev video if needed */
		rev = req;
		(*term.t_rev)(req);

		/* scan through the line and dump it to the screen and
		   the virtual screen array				*/
		cp3 = &vp1->v_text[vissize];
		while (cp1 < cp3 ) {
			TTputc(*cp1);
			++ttcol;
			*cp2++ = *cp1++;
		}
		if ( eolexist && (!req) )
			if (ttcol<term.t_ncol) TTeeol();
		else
			while(ttcol<term.t_ncol)
			{
				TTputc(' ');
				++ttcol;
			}
		;
		
		/* turn rev video off */

		if (rev)
			(*term.t_rev)(FALSE);

		/* update the needed flags */
		vp1->v_flag &= ~VFCHG;
#if HARDSCRL
		vp2->v_flag &= ~VFCHG;
#endif
		vp2->v_size = vissize;
		if (req)
			vp1->v_flag |= VFREV;
		else
			vp1->v_flag &= ~VFREV;
#if	COLOR
		vp1->v_fcolor = vp1->v_rfcolor;
		vp1->v_bcolor = vp1->v_rbcolor;
#endif
		return(TRUE);
	}
#endif

	/* Try to perform a mininmal update of the line */
	
	upcol = 0;
	ttcol = -1;
	
	cp5 = &vp1->v_text[vissize];
	
	/* Chop any trailing blanks */

	while((vissize>1) && cp5[-1] == ' ')
	{
		vissize--;
		cp5--;
	}

	cp4 = &vp2->v_text[vissize > vp2->v_size ? vissize : vp2->v_size];

	/* Pad the physical screen line out to the same length as the virtual line */

	cp3 = &vp2->v_text[vp2->v_size];

	while(cp3<cp4)
		*cp3++ = ' ';

	/* cp1 = pointer to physical screen, cp2 = pointer to virtual screen */
	/* cp3 = pointer to start of a matching segment in physical screen */

#if REVSTA
	TTrev(rev);
#endif

	while(cp1 != cp5)
	{
		/* Skip over a matching sequence of characters */
	
		cp3 = cp1;
		overlap = 0;
		while( *cp1 == *cp2 && cp1 != cp5)
		{
			cp1++; cp2++; upcol++; overlap++;
		}

		/* If it's not worth skipping these with a move, print them out */
	
		if(overlap < 5 && ttcol != -1 )
		{
			while(cp3!=cp1)
			{
				TTputc(*cp3++);
				ttcol++;
			}
		}
		/* Now output the unmatching characters */

		while( *cp1 != *cp2 && cp1 != cp5 )
		{
			if ( ttcol != upcol ) movecursor(row,upcol);
			TTputc(*cp2++ = *cp1++);
			upcol++;
			ttcol++;
		}
	}

	/* Output the remaining characters in the virtual line */

	while(cp1 != cp5)
	{
		if ( ttcol != upcol ) movecursor(row,upcol);
		TTputc(*cp2++ = *cp1++);
		upcol++;
		ttcol++;
	}

	/* If there are still characters in the physical line, remove them */

	if(cp2 != cp4)
	{
		if ( ttcol != upcol ) movecursor(row,upcol);
		if((!req) && eolexist && (vp2->v_size - upcol) > 3 )
			TTeeol();
		else
			while(cp2 != cp4)
			{
				cp2++;
				TTputc(' ');
				ttcol++;
			}
		;
	}

#if	REVSTA
	TTrev(FALSE);
#endif
	vp1->v_flag &= ~VFCHG;		/* flag this line as updated */
	vp1->v_size = vp2->v_size = vissize;
	TTflush();
	return(TRUE);
#endif
}
#endif

/*
 * Redisplay the mode line for the window pointed to by the "wp". This is the
 * only routine that has any idea of how the modeline is formatted. You can
 * change the modeline format by hacking at this routine. Called by "update"
 * any time there is a dirty window.
 */
#if	LANCASTER
VOID PASCAL NEAR modeline(wp)
#else
PASCAL NEAR modeline(wp)
#endif

WINDOW *wp;	/* window to update modeline for */

{
	register char *cp;
	register int c;
	register int n;		/* cursor position count */
	register BUFFER *bp;
	register int i;		/* loop index */
	register int lchar; 	/* character to draw line in buffer with */
	register int firstm;	/* is this the first mode? */
	char tline[NLINE];		/* buffer for part of mode line */

	/* don't bother if there is none! */
	if (modeflag == FALSE)
		return;

	n = wp->w_toprow+wp->w_ntrows;		/* Location. */
#if	COLOR
	vscreen[n]->v_flag |= VFCHG | VFCOL;		/* Redraw next time. */
	vscreen[n]->v_rfcolor = 0;			/* black on */
	vscreen[n]->v_rbcolor = 7;			/* white.....*/
#else
	vscreen[n]->v_flag |= VFCHG | VFREQ | VFCOL;	/* Redraw next time. */
#endif
	vtmove(n, 0);				/* Seek to right line. */
	if (wp == curwp)			/* mark the current buffer */
		lchar = '=';
	else
#if	REVSTA
	if (revexist)
		lchar = ' ';
	else
#endif
		lchar = '-';

	bp = wp->w_bufp;
	if ((bp->b_flag&BFTRUNC) != 0)		/* "#" if truncated */
		vtputc('#',1);
	else
		vtputc(lchar,1);

	if ((bp->b_flag&BFCHG) != 0)		/* "*" if changed. */
		vtputc('*',1);
	else
		vtputc(lchar,1);

	if ((bp->b_flag&BFNAROW) != 0) {		/* "<>" if narrowed */
		vtputc('<',1);
		vtputc('>',1);
	} else {
		vtputc(lchar,1);
		vtputc(lchar,1);
	}

	n  = 4;
	strcpy(tline, " "); 			/* Buffer name. */
	strcat(tline, PROGNAME);
	strcat(tline, " ");
#if	LANCASTER
	strcat(tline, version_string);
#else
	strcat(tline, VERSION);
#endif
	strcat(tline, " ");

	/* are we horizontally scrolled? */
	if (wp->w_fcol > 0) {
		strcat(tline, "[<");
		strcat(tline, int_asc(wp->w_fcol));
		strcat(tline, "]");
	}

	/* display the modes */
	strcat(tline, "(");
	firstm = TRUE;
	for (i = 0; i < NUMMODES; i++)	/* add in the mode flags */
		if (wp->w_bufp->b_mode & (1 << i)) {
			if (firstm != TRUE)
				strcat(tline, " ");
			firstm = FALSE;
			strcat(tline, modename[i]);
		}
	strcat(tline,") ");

	cp = &tline[0];
	while ((c = *cp++) != 0) {
		vtputc(c,1);
		++n;
	}

#if 0
	vtputc(lchar,1);
	vtputc((wp->w_flag&WFCOLR) != 0  ? 'C' : lchar,1);
	vtputc((wp->w_flag&WFMODE) != 0  ? 'M' : lchar,1);
	vtputc((wp->w_flag&WFHARD) != 0  ? 'H' : lchar,1);
	vtputc((wp->w_flag&WFEDIT) != 0  ? 'E' : lchar,1);
	vtputc((wp->w_flag&WFMOVE) != 0  ? 'V' : lchar,1);
	vtputc((wp->w_flag&WFFORCE) != 0 ? 'F' : lchar,1);
	vtputc(lchar,1);
	n += 8;
#endif

	vtputc(lchar,1);
	vtputc(lchar,1);
	vtputc(' ',1);
	n += 3;
	cp = &bp->b_bname[0];

	while ((c = *cp++) != 0) {
		vtputc(c,1);
		++n;
	}

	vtputc(' ',1);
	vtputc(lchar,1);
	vtputc(lchar,1);
	n += 3;

	if (bp->b_fname[0] != 0) {	/* File name. */
		vtputc(' ',1);
		++n;
		cp = TEXT34;
/*                   "File: " */

		while ((c = *cp++) != 0) {
			vtputc(c,1);
			++n;
		}

		cp = &bp->b_fname[0];

		while ((c = *cp++) != 0) {
			vtputc(c,1);
			++n;
	        }

		vtputc(' ',1);
		++n;
	}

#if	!TERMCAP
	/* Write out any "=" signs to the end of the line */

	if ( lchar == '=' )
#endif
	while (n < term.t_ncol) {	/* Pad to full width. */
		vtputc(lchar,1);
		++n;
	}

	vteeol();
}

PASCAL NEAR upmode()	/* update all the mode lines */

{
	register WINDOW *wp;

	wp = wheadp;
	while (wp != NULL) {
		wp->w_flag |= WFMODE;
		wp = wp->w_wndp;
	}
}

PASCAL NEAR upwind()	/* force hard updates on all windows */

{
	register WINDOW *wp;

	wp = wheadp;
	while (wp != NULL) {
		wp->w_flag |= WFHARD|WFMODE;
		wp = wp->w_wndp;
	}
}

/*
 * Send a command to the terminal to move the hardware cursor to row "row"
 * and column "col". The row and column arguments are origin 0. Optimize out
 * random calls. Update "ttrow" and "ttcol".
 */
PASCAL NEAR movecursor(row, col)
    {
    if (row!=ttrow || col!=ttcol)
	{
	ttrow = row;
	ttcol = col;
	TTmove(row, col);
	}
    }

/*
 * Erase the message line. This is a special routine because the message line
 * is not considered to be part of the virtual screen. It always works
 * immediately; the terminal buffer is flushed via a call to the flusher.
 */
#if LANCASTER
VOID PASCAL NEAR mlerase()
#else
PASCAL NEAR mlerase()
#endif
    {
    int i;
    
    movecursor(term.t_nrow, 0);
    if (discmd == FALSE)
	return;

#if	COLOR
     TTforg(7);
     TTbacg(0);
#endif
    if (eolexist == TRUE)
	    TTeeol();
    else {
	for (i = 0; i < term.t_ncol - 1; i++)
	    TTputc(' ');
	movecursor(term.t_nrow, 1);	/* force the move! */
	movecursor(term.t_nrow, 0);
    }
    TTflush();
    mpresf = FALSE;
    }

/*
 * Write a message into the message line. Keep track of the physical cursor
 * position. A small class of printf like format items is handled. Assumes the
 * stack grows down; this assumption is made by the "+=" in the argument scan
 * loop. If  STACK_GROWS_UP  is set in estruct.h, then we'll assume that the
 * stack grows up and use "-=" instead of "+=". Set the "message line"
 *  flag TRUE.  Don't write beyond the end of the current terminal width.
 */

PASCAL NEAR mlout(c)

int c;	/* character to write on modeline */

{
	if (ttcol + 1 < term.t_ncol)
		TTputc(c);
	*lastptr++ = c;
}

#if	STACK_GROWS_UP
#define	ADJUST(ptr, dtype)	ptr -= sizeof(dtype)
#else
#define	ADJUST(ptr, dtype)	ptr += sizeof(dtype)
#endif

#if	LANCASTER
VOID CDECL NEAR mlwrite(fmt, arg)
#else
CDECL NEAR mlwrite(fmt, arg)
#endif

char *fmt;	/* format string for output */
char *arg;	/* pointer to first argument to print */

{
	register int c; 	/* current char in format string */
	register char *ap;	/* ptr to current data field */

	/* if we are not currently echoing on the command line, abort this */
	if (discmd == FALSE)
		return;

#if	COLOR
	/* set up the proper colors for the command line */
	TTforg(7);
	TTbacg(0);
#endif

	/* if we can not erase to end-of-line, do it manually */
	if (eolexist == FALSE) {
		mlerase();
		TTflush();
	}

	movecursor(term.t_nrow, 0);
 	lastptr = &lastmesg[0];		/* setup to record message */
	ap = (char *) &arg;
	while ((c = *fmt++) != 0) {
		if (c != '%') {
			mlout(c);
			++ttcol;
		} else {
			c = *fmt++;
			switch (c) {
				case 'd':
					mlputi(*(int *)ap, 10);
			                ADJUST(ap, int);
					break;

				case 'o':
					mlputi(*(int *)ap,  8);
					ADJUST(ap, int);
					break;

				case 'x':
					mlputi(*(int *)ap, 16);
					ADJUST(ap, int);
					break;

				case 'D':
					mlputli(*(long *)ap, 10);
					ADJUST(ap, long);
					break;

				case 's':
					mlputs(*(char **)ap);
					ADJUST(ap, char *);
					break;

				case 'f':
					mlputf(*(int *)ap);
					ADJUST(ap, int);
					break;

				default:
					mlout(c);
					++ttcol;
			}
		}
	}

	/* if we can, erase to the end of screen */
	if (eolexist == TRUE)
		TTeeol();
	TTflush();
	mpresf = TRUE;
	*lastptr = 0;	/* terminate lastmesg[] */
}

/*	Force a string out to the message line regardless of the
	current $discmd setting. This is needed when $debug is TRUE
	and for the write-message and clear-message-line commands
*/

PASCAL NEAR mlforce(s)

char *s;	/* string to force out */

{
	register int oldcmd;	/* original command display flag */

	oldcmd = discmd;	/* save the discmd value */
	discmd = TRUE;		/* and turn display on */
	mlwrite(s);		/* write the string out */
	discmd = oldcmd;	/* and restore the original setting */
}

/*
 * Write out a string. Update the physical cursor position. This assumes that
 * the characters in the string all have width "1"; if this is not the case
 * things will get screwed up a little.
 */
PASCAL NEAR mlputs(s)
    char *s;
    {
    register int c;

    while ((c = *s++) != 0)
	{
	mlout(c);
	++ttcol;
	}
    }

/*
 * Write out an integer, in the specified radix. Update the physical cursor
 * position.
 */
PASCAL NEAR mlputi(i, r)
    {
    register int q;
    static char hexdigits[] = "0123456789ABCDEF";

    if (i < 0)
	{
	i = -i;
	mlout('-');
	}

    q = i/r;

    if (q != 0)
	mlputi(q, r);

    mlout(hexdigits[i%r]);
    ++ttcol;
    }

/*
 * do the same except as a long integer.
 */
PASCAL NEAR mlputli(l, r)
    long l;
    {
    register long q;

    if (l < 0)
	{
	l = -l;
	mlout('-');
	}

    q = l/r;

    if (q != 0)
	mlputli(q, r);

    mlout((int)(l%r)+'0');
    ++ttcol;
    }

/*
 *	write out a scaled integer with two decimal places
 */

PASCAL NEAR mlputf(s)

int s;	/* scaled integer to output */

{
	int i;	/* integer portion of number */
	int f;	/* fractional portion of number */

	/* break it up */
	i = s / 100;
	f = s % 100;

	/* send out the integer portion */
	mlputi(i, 10);
	mlout('.');
	mlout((f / 10) + '0');
	mlout((f % 10) + '0');
	ttcol += 3;
}       

#if RAINBOW

PASCAL NEAR putline(row, col, buf)
    int row, col;
    char buf[];
    {
    int n;

    n = strlen(buf);
    if (col + n - 1 > term.t_ncol)
	n = term.t_ncol - col + 1;
    Put_Data(row, col, n, buf);
    }
#endif

